# costume.py
"""
========================================
🎮 COSTUMEモジュール
========================================

pygame-zeroとpgzhelperのActorクラスに高機能なアニメーション機能を追加するモジュールです。

📋 使用方法:
    from costume import *
    
    player = Actor('player.png')
    player.images = ['walk1.png', 'walk2.png', 'walk3.png']
    player.fps = 8
    player.animate()  # ゲームループ内で呼び出し

========================================
📂 プロパティ
========================================

🔹 actor.images
    型: リスト
    説明: アニメーションで使用する画像のリスト
    例: player.images = ['walk1.png', 'walk2.png', 'walk3.png']

🔹 actor.fps
    型: 数値
    説明: 1秒間に何回画像を切り替えるかの速度
    例: player.fps = 8  # 1秒間に8回切り替え

========================================
🛠️ 基本メソッド
========================================

🔸 actor.animate()
    説明: 自動的にコスチュームを順番に切り替える
    用途: ゲームループ内で継続的なアニメーション
    例: player.animate()

🔸 actor.change_costume()
    説明: 手動で次のコスチュームに1つ進める
    用途: ボタン押下やイベント発生時
    例: player.change_costume()

🔸 actor.set_costumes(images, fps=5)
    説明: 画像リストと速度を一度に設定
    パラメータ: images=画像リスト, fps=速度
    例: player.set_costumes(['walk1.png', 'walk2.png'], fps=6)

========================================
⏯️ 再生制御メソッド
========================================

🔸 actor.play()
    説明: アニメーションを開始/再開する
    例: player.play()

🔸 actor.pause()
    説明: アニメーションを一時停止する
    例: player.pause()

🔸 actor.resume()
    説明: 一時停止したアニメーションを再開する
    例: player.resume()

🔸 actor.stop()
    説明: アニメーションを停止し、最初の画像に戻る
    例: player.stop()

========================================
🎮 高度な制御メソッド
========================================

🔸 actor.jump_to_costume(index)
    説明: 特定の画像番号に直接ジャンプ
    パラメータ: index=画像番号（0から始まる）
    戻り値: 成功時True、失敗時False
    例: player.jump_to_costume(2)  # 3番目の画像

🔸 actor.set_reverse(reverse=True)
    説明: 逆再生を設定する
    パラメータ: reverse=True（逆再生）/False（通常再生）
    例: player.set_reverse(True)

🔸 actor.set_loop(loop=True)
    説明: ループ再生を設定する
    パラメータ: loop=True（ループ）/False（1回のみ）
    例: explosion.set_loop(False)  # 1回だけ再生

🔸 actor.set_fps(fps)
    説明: アニメーション速度を変更する
    例: player.set_fps(10)

🔸 actor.set_animation_range(start_index, end_index=None)
    説明: アニメーションする範囲を設定する
    パラメータ: start_index=開始番号, end_index=終了番号
    例: player.set_animation_range(0, 2)  # 0〜2番目でアニメーション

🔸 actor.reset_animation_range()
    説明: アニメーション範囲をリセット（全画像を使用）
    例: player.reset_animation_range()

========================================
🎯 コスチューム設定メソッド
========================================

🔸 actor.set_costume_by_name(image_name)
    説明: 画像名でコスチュームを設定する
    パラメータ: image_name=画像ファイル名
    戻り値: 成功時True、失敗時False
    例: player.set_costume_by_name('attack.png')

🔸 actor.set_costume_by_number(costume_number)
    説明: コスチューム番号でコスチュームを設定する（0から始まる）
    パラメータ: costume_number=コスチューム番号
    戻り値: 成功時True、失敗時False
    例: player.set_costume_by_number(0)  # 最初の画像

🔸 actor.set_costume_random()
    説明: ランダムにコスチュームを選択する
    戻り値: 成功時True、失敗時False
    例: player.set_costume_random()

🔸 actor.set_costume_random_range(start_number, end_number)
    説明: 指定範囲でランダムにコスチュームを選択する（0から始まる）
    パラメータ: start_number=開始番号, end_number=終了番号
    戻り値: 成功時True、失敗時False
    例: player.set_costume_random_range(1, 3)  # 1〜3番目からランダム

🔸 actor.next_costume()
    説明: 次のコスチュームに変更（アニメーション範囲を考慮）
    戻り値: 成功時True、失敗時False
    例: player.next_costume()

🔸 actor.previous_costume()
    説明: 前のコスチュームに変更（アニメーション範囲を考慮）
    戻り値: 成功時True、失敗時False
    例: player.previous_costume()

========================================
📊 情報取得メソッド
========================================

🔸 actor.get_current_image()
    説明: 現在表示中の画像を取得
    戻り値: 画像オブジェクト
    例: current_img = player.get_current_image()

🔸 actor.get_image_at(index)
    説明: 指定番号の画像を取得
    パラメータ: index=画像番号
    戻り値: 画像オブジェクト
    例: first_img = player.get_image_at(0)

🔸 actor.get_current_index()
    説明: 現在の画像番号を取得（0から始まる）
    戻り値: 画像番号
    例: current_num = player.get_current_index()

🔸 actor.get_costume_number()
    説明: 現在のコスチューム番号を取得（get_current_index()と同じ）
    戻り値: コスチューム番号（0から始まる）
    例: costume_num = player.get_costume_number()

🔸 actor.get_total_images()
    説明: 総画像数を取得
    戻り値: 画像の総数
    例: total = player.get_total_images()

🔸 actor.get_total_costumes()
    説明: 総コスチューム数を取得（get_total_images()と同じ）
    戻り値: コスチュームの総数
    例: total = player.get_total_costumes()

🔸 actor.get_animation_range()
    説明: 現在のアニメーション範囲を取得
    戻り値: (開始番号, 終了番号)のタプル
    例: start, end = player.get_animation_range()

========================================
🔍 状態確認メソッド
========================================

🔸 actor.is_playing()
    説明: アニメーションが再生中かを確認
    戻り値: True=再生中、False=停止中
    例: if player.is_playing(): print("再生中")

🔸 actor.is_paused()
    説明: アニメーションが一時停止中かを確認
    戻り値: True=一時停止中、False=再生中または停止中
    例: if player.is_paused(): print("一時停止中")

🔸 actor.is_animation_completed()
    説明: アニメーションが完了したかを確認（ループ無効時）
    戻り値: True=完了、False=未完了
    例: if explosion.is_animation_completed(): effects.remove(explosion)

========================================
🎪 実践例
========================================

# 基本的なアニメーション
player = Actor('player.png')
player.images = ['walk1.png', 'walk2.png', 'walk3.png']
player.fps = 8

def update():
    player.animate()

# 状態別アニメーション
def update_player():
    if player.is_attacking:
        if not player.is_playing():
            player.set_costumes(['attack1.png', 'attack2.png'], fps=15)
            player.set_loop(False)
            player.play()
        if player.is_animation_completed():
            player.is_attacking = False
            player.set_costumes(['idle1.png', 'idle2.png'], fps=4)
            player.play()
    elif player.is_moving:
        player.set_costumes(['walk1.png', 'walk2.png', 'walk3.png'], fps=8)
        player.animate()

# 範囲指定アニメーション
enemy.images = ['idle1', 'idle2', 'walk1', 'walk2', 'walk3', 'attack1', 'attack2']
enemy.set_animation_range(0, 1)  # idle1〜idle2でアニメーション
enemy.animate()

# ランダムコスチューム
dice.images = ['dice1', 'dice2', 'dice3', 'dice4', 'dice5', 'dice6']
dice.set_costume_random_range(0, 5)  # 全てからランダム

========================================
🔧 互換性
========================================

- pygame-zero標準Actor: 基本機能のみ（flip_x, flip_yなし）
- pgzhelperのActor: 完全対応（flip_x, flip_y対応）

========================================
"""
import time

class CostumeManager:
    def __init__(self):
        self.actors = {}  # アクターごとの情報を格納する辞書
    
    def _ensure_actor_data(self, actor):
        """アクターのデータが存在することを確認し、なければ初期化"""
        if actor not in self.actors:
            # アクターにimagesとfpsが設定されているかチェック
            images = getattr(actor, 'images', [])
            fps = getattr(actor, 'fps', 5)
            
            self.actors[actor] = {
                'images': images,
                'current': 0,
                'fps': fps,
                'last_time': time.time(),
                'is_playing': True,
                'is_paused': False,
                'reverse': False,
                'loop': True,
                'animation_completed': False,
                'start_index': 0,        # アニメーション開始インデックス
                'end_index': None        # アニメーション終了インデックス（Noneなら最後まで）
            }
    
    def _update_costume_index(self, actor):
        """コスチュームインデックスを更新し、画像を変更する"""
        self._ensure_actor_data(actor)
        info = self.actors[actor]
        current = info['current']
        
        # 最新のimagesとfpsをアクターから取得
        images = getattr(actor, 'images', [])
        fps = getattr(actor, 'fps', 5)
        
        # 情報を更新
        info['images'] = images
        info['fps'] = fps
        
        if not images:  # 画像リストが空の場合は何もしない
            return
        
        # アニメーション範囲を決定
        start_idx = info['start_index']
        end_idx = info['end_index'] if info['end_index'] is not None else len(images) - 1
        
        # pgzhelperの場合のみ向きを保存（pygame-zeroは不要）
        need_preserve_orientation = hasattr(actor, 'flip_x') and hasattr(actor, 'flip_y')
        if need_preserve_orientation:
            current_angle = getattr(actor, 'angle', 0)
            current_flip_x = getattr(actor, 'flip_x', False)
            current_flip_y = getattr(actor, 'flip_y', False)
        
        # 次のインデックスを計算
        if info['reverse']:
            next_index = current - 1
            if next_index < start_idx:
                if info['loop']:
                    next_index = end_idx
                else:
                    next_index = start_idx
                    info['animation_completed'] = True
                    info['is_playing'] = False
        else:
            next_index = current + 1
            if next_index > end_idx:
                if info['loop']:
                    next_index = start_idx
                else:
                    next_index = end_idx
                    info['animation_completed'] = True
                    info['is_playing'] = False
        
        # 画像を変更
        actor.image = images[next_index]
        
        # pgzhelperの場合のみ向きを復元
        if need_preserve_orientation:
            actor.angle = current_angle
            actor.flip_x = current_flip_x
            actor.flip_y = current_flip_y
        
        # 現在のインデックスを更新
        info['current'] = next_index

# グローバルなCostumeManagerインスタンス
_costume_manager = CostumeManager()

# Actor クラスの拡張メソッド
def actor_set_costumes(self, images, fps=5):
    """アクターのコスチューム（画像リスト）を設定する"""
    self.images = images
    self.fps = fps
    _costume_manager._ensure_actor_data(self)
    _costume_manager.actors[self]['images'] = images
    _costume_manager.actors[self]['fps'] = fps
    _costume_manager.actors[self]['animation_completed'] = False

def actor_animate(self):
    """アニメーションを実行（自動的に適切なタイミングで画像を切り替える）"""
    _costume_manager._ensure_actor_data(self)
    info = _costume_manager.actors[self]
    
    # 停止中または一時停止中は何もしない
    if not info['is_playing'] or info['is_paused']:
        return
        
    current_time = time.time()
    time_diff = current_time - info['last_time']
    
    # 最新のfpsを取得
    fps = getattr(self, 'fps', 5)
    
    # FPSに基づいてアニメーションのタイミングを計算
    if time_diff >= 1.0 / fps:
        _costume_manager._update_costume_index(self)
        info['last_time'] = current_time

def actor_change_costume(self):
    """次のコスチュームに変更する（手動で呼び出す場合）"""
    _costume_manager._ensure_actor_data(self)
    _costume_manager._update_costume_index(self)

def actor_jump_to_costume(self, index):
    """特定の画像番号にジャンプする"""
    _costume_manager._ensure_actor_data(self)
    images = getattr(self, 'images', [])
    if 0 <= index < len(images):
        # pgzhelperの場合のみ向きを保存・復元
        need_preserve_orientation = hasattr(self, 'flip_x') and hasattr(self, 'flip_y')
        if need_preserve_orientation:
            current_angle = getattr(self, 'angle', 0)
            current_flip_x = getattr(self, 'flip_x', False)
            current_flip_y = getattr(self, 'flip_y', False)
        
        # 画像を変更
        self.image = images[index]
        _costume_manager.actors[self]['current'] = index
        
        # pgzhelperの場合のみ向きを復元
        if need_preserve_orientation:
            self.angle = current_angle
            self.flip_x = current_flip_x
            self.flip_y = current_flip_y
        
        return True
    return False

def actor_play(self):
    """アニメーションを開始/再開する"""
    _costume_manager._ensure_actor_data(self)
    _costume_manager.actors[self]['is_playing'] = True
    _costume_manager.actors[self]['is_paused'] = False
    _costume_manager.actors[self]['animation_completed'] = False

def actor_stop(self):
    """アニメーションを停止し、最初の画像に戻る"""
    _costume_manager._ensure_actor_data(self)
    _costume_manager.actors[self]['is_playing'] = False
    _costume_manager.actors[self]['is_paused'] = False
    _costume_manager.actors[self]['animation_completed'] = False
    self.jump_to_costume(0)

def actor_pause(self):
    """アニメーションを一時停止する"""
    _costume_manager._ensure_actor_data(self)
    _costume_manager.actors[self]['is_paused'] = True

def actor_resume(self):
    """一時停止したアニメーションを再開する"""
    _costume_manager._ensure_actor_data(self)
    _costume_manager.actors[self]['is_paused'] = False

def actor_set_reverse(self, reverse=True):
    """逆再生を設定する"""
    _costume_manager._ensure_actor_data(self)
    _costume_manager.actors[self]['reverse'] = reverse

def actor_set_loop(self, loop=True):
    """ループ再生を設定する"""
    _costume_manager._ensure_actor_data(self)
    _costume_manager.actors[self]['loop'] = loop

def actor_is_animation_completed(self):
    """アニメーションが完了したかを検知する"""
    _costume_manager._ensure_actor_data(self)
    return _costume_manager.actors[self]['animation_completed']

def actor_get_current_image(self):
    """現在の画像を取得する"""
    _costume_manager._ensure_actor_data(self)
    images = getattr(self, 'images', [])
    current = _costume_manager.actors[self]['current']
    if images and 0 <= current < len(images):
        return images[current]
    return None

def actor_get_image_at(self, index):
    """指定番号の画像を取得する"""
    images = getattr(self, 'images', [])
    if 0 <= index < len(images):
        return images[index]
    return None

def actor_get_current_index(self):
    """現在の画像番号を取得する"""
    _costume_manager._ensure_actor_data(self)
    return _costume_manager.actors[self]['current']

def actor_get_total_images(self):
    """総画像数を取得する"""
    images = getattr(self, 'images', [])
    return len(images)

def actor_is_playing(self):
    """アニメーションが再生中かを確認する"""
    _costume_manager._ensure_actor_data(self)
    info = _costume_manager.actors[self]
    return info['is_playing'] and not info['is_paused']

def actor_is_paused(self):
    """アニメーションが一時停止中かを確認する"""
    _costume_manager._ensure_actor_data(self)
    return _costume_manager.actors[self]['is_paused']

def actor_set_fps(self, fps):
    """アニメーション速度を変更する"""
    self.fps = fps
    _costume_manager._ensure_actor_data(self)
    _costume_manager.actors[self]['fps'] = fps

def actor_set_animation_range(self, start_index, end_index=None):
    """アニメーションする範囲を設定する"""
    _costume_manager._ensure_actor_data(self)
    images = getattr(self, 'images', [])
    
    # 範囲チェック
    if start_index < 0:
        start_index = 0
    if end_index is None:
        end_index = len(images) - 1
    elif end_index >= len(images):
        end_index = len(images) - 1
    
    _costume_manager.actors[self]['start_index'] = start_index
    _costume_manager.actors[self]['end_index'] = end_index
    
    # 現在のインデックスが範囲外なら調整
    current = _costume_manager.actors[self]['current']
    if current < start_index or current > end_index:
        _costume_manager.actors[self]['current'] = start_index
        self.image = images[start_index]

def actor_reset_animation_range(self):
    """アニメーション範囲をリセット（全画像を使用）"""
    _costume_manager._ensure_actor_data(self)
    images = getattr(self, 'images', [])
    _costume_manager.actors[self]['start_index'] = 0
    _costume_manager.actors[self]['end_index'] = len(images) - 1 if images else None

def actor_set_costume_by_name(self, image_name):
    """画像名でコスチュームを設定する（Scratchの「コスチュームを○にする」）"""
    images = getattr(self, 'images', [])
    
    for i, img in enumerate(images):
        # 画像名が文字列の場合
        if isinstance(img, str) and img == image_name:
            return self.jump_to_costume(i)
        # 画像ファイル名で比較
        elif hasattr(img, 'name') and img.name == image_name:
            return self.jump_to_costume(i)
    
    return False

def actor_set_costume_random(self):
    """ランダムにコスチュームを選択する"""
    import random
    images = getattr(self, 'images', [])
    
    if images:
        random_index = random.randint(0, len(images) - 1)
        return self.jump_to_costume(random_index)
    return False

def actor_next_costume(self):
    """次のコスチュームに変更（範囲を考慮）"""
    _costume_manager._ensure_actor_data(self)
    info = _costume_manager.actors[self]
    images = getattr(self, 'images', [])
    
    if not images:
        return False
    
    current = info['current']
    start_idx = info['start_index']
    end_idx = info['end_index'] if info['end_index'] is not None else len(images) - 1
    
    next_index = current + 1
    if next_index > end_idx:
        next_index = start_idx  # 範囲の最初に戻る
    
    return self.jump_to_costume(next_index)

def actor_previous_costume(self):
    """前のコスチュームに変更（範囲を考慮）"""
    _costume_manager._ensure_actor_data(self)
    info = _costume_manager.actors[self]
    images = getattr(self, 'images', [])
    
    if not images:
        return False
    
    current = info['current']
    start_idx = info['start_index']
    end_idx = info['end_index'] if info['end_index'] is not None else len(images) - 1
    
    prev_index = current - 1
    if prev_index < start_idx:
        prev_index = end_idx  # 範囲の最後に移動
    
    return self.jump_to_costume(prev_index)

def actor_set_costume_by_number(self, costume_number):
    """コスチューム番号でコスチュームを設定する（Scratchの番号は1から始まる）"""
    images = getattr(self, 'images', [])
    
    if images and 1 <= costume_number <= len(images):
        # Scratchは1から始まるので、1を引いてインデックスに変換
        index = costume_number - 1
        return self.jump_to_costume(index)
    return False

def actor_set_costume_random_range(self, start_number, end_number):
    """指定範囲でランダムにコスチュームを選択する（Scratchの「コスチュームを○から○までの乱数にする」）"""
    import random
    images = getattr(self, 'images', [])
    
    if not images:
        return False
    
    # 範囲チェック（Scratchは1から始まる）
    max_costume = len(images)
    start_number = max(1, min(start_number, max_costume))
    end_number = max(1, min(end_number, max_costume))
    
    # start_numberの方が大きい場合は入れ替える
    if start_number > end_number:
        start_number, end_number = end_number, start_number
    
    # ランダムな番号を生成（Scratchの番号）
    random_number = random.randint(start_number, end_number)
    
    # インデックスに変換して設定
    return self.set_costume_by_number(random_number)

def actor_get_costume_number(self):
    """現在のコスチューム番号を取得する（Scratchライク：1から始まる）"""
    current_index = self.get_current_index()
    return current_index + 1  # Scratchは1から始まるので1を足す

def actor_get_animation_range(self):
    """現在のアニメーション範囲を取得する"""
    _costume_manager._ensure_actor_data(self)
    info = _costume_manager.actors[self]
    start_idx = info['start_index']
    end_idx = info['end_index']
    images = getattr(self, 'images', [])
    
    if end_idx is None and images:
        end_idx = len(images) - 1
    
    return (start_idx, end_idx)

def actor_get_total_costumes(self):
    """総コスチューム数を取得する（actor.get_total_images()のエイリアス）"""
    return self.get_total_images()

# すべてのActorクラスにメソッドを追加する関数
def enhance_actor_classes():
    """利用可能なすべてのActorクラスにコスチューム機能を追加する"""
    actor_classes = []
    
    # pygame-zeroの標準Actorを探す
    try:
        import pgzero.actor
        actor_classes.append(pgzero.actor.Actor)
        print("pygame-zero Actor found and enhanced!")
    except ImportError:
        pass
    
    # pgzhelperのActorを探す
    try:
        import pgzhelper
        actor_classes.append(pgzhelper.Actor)
        print("pgzhelper Actor found and enhanced!")
    except ImportError:
        pass
    
    # 見つからない場合は、グローバル名前空間のActorを探す
    if not actor_classes:
        try:
            import builtins
            if hasattr(builtins, 'Actor'):
                actor_classes.append(builtins.Actor)
                print("Global Actor found and enhanced!")
        except:
            pass
    
    # 各Actorクラスにメソッドを追加
    for ActorClass in actor_classes:
        # メソッドを追加
        ActorClass.set_costumes = actor_set_costumes
        ActorClass.animate = actor_animate
        ActorClass.change_costume = actor_change_costume
        ActorClass.jump_to_costume = actor_jump_to_costume
        ActorClass.play = actor_play
        ActorClass.stop = actor_stop
        ActorClass.pause = actor_pause
        ActorClass.resume = actor_resume
        ActorClass.set_reverse = actor_set_reverse
        ActorClass.set_loop = actor_set_loop
        ActorClass.is_animation_completed = actor_is_animation_completed
        ActorClass.get_current_image = actor_get_current_image
        ActorClass.get_image_at = actor_get_image_at
        ActorClass.get_current_index = actor_get_current_index
        ActorClass.get_total_images = actor_get_total_images
        ActorClass.is_playing = actor_is_playing
        ActorClass.is_paused = actor_is_paused
        ActorClass.set_fps = actor_set_fps
        
        # メソッドを追加
        ActorClass.set_costumes = actor_set_costumes
        ActorClass.animate = actor_animate
        ActorClass.change_costume = actor_change_costume
        ActorClass.jump_to_costume = actor_jump_to_costume
        ActorClass.play = actor_play
        ActorClass.stop = actor_stop
        ActorClass.pause = actor_pause
        ActorClass.resume = actor_resume
        ActorClass.set_reverse = actor_set_reverse
        ActorClass.set_loop = actor_set_loop
        ActorClass.is_animation_completed = actor_is_animation_completed
        ActorClass.get_current_image = actor_get_current_image
        ActorClass.get_image_at = actor_get_image_at
        ActorClass.get_current_index = actor_get_current_index
        ActorClass.get_total_images = actor_get_total_images
        ActorClass.is_playing = actor_is_playing
        ActorClass.is_paused = actor_is_paused
        ActorClass.set_fps = actor_set_fps
        
        # 新機能メソッドを追加
        ActorClass.set_animation_range = actor_set_animation_range
        ActorClass.reset_animation_range = actor_reset_animation_range
        ActorClass.set_costume_by_name = actor_set_costume_by_name
        ActorClass.set_costume_by_number = actor_set_costume_by_number
        ActorClass.set_costume_random = actor_set_costume_random
        ActorClass.set_costume_random_range = actor_set_costume_random_range
        ActorClass.next_costume = actor_next_costume
        ActorClass.previous_costume = actor_previous_costume
        ActorClass.get_animation_range = actor_get_animation_range
        ActorClass.get_costume_number = actor_get_costume_number
        ActorClass.get_total_costumes = actor_get_total_costumes
        
        # imagesとfpsプロパティをデフォルトで追加
        if not hasattr(ActorClass, 'images'):
            ActorClass.images = []
        if not hasattr(ActorClass, 'fps'):
            ActorClass.fps = 5

# 後から呼び出せるよう、グローバル関数として定義
def patch_actor_in_globals():
    """グローバル名前空間のActorクラスを拡張"""
    import sys
    frame = sys._getframe(1)  # 呼び出し元のフレーム
    
    if 'Actor' in frame.f_globals:
        ActorClass = frame.f_globals['Actor']
        
        # メソッドを追加
        ActorClass.set_costumes = actor_set_costumes
        ActorClass.animate = actor_animate
        ActorClass.change_costume = actor_change_costume
        ActorClass.jump_to_costume = actor_jump_to_costume
        ActorClass.play = actor_play
        ActorClass.stop = actor_stop
        ActorClass.pause = actor_pause
        ActorClass.resume = actor_resume
        ActorClass.set_reverse = actor_set_reverse
        ActorClass.set_loop = actor_set_loop
        ActorClass.is_animation_completed = actor_is_animation_completed
        ActorClass.get_current_image = actor_get_current_image
        ActorClass.get_image_at = actor_get_image_at
        ActorClass.get_current_index = actor_get_current_index
        ActorClass.get_total_images = actor_get_total_images
        ActorClass.is_playing = actor_is_playing
        ActorClass.is_paused = actor_is_paused
        ActorClass.set_fps = actor_set_fps
        
        print("Global Actor class enhanced with costume functionality!")
        return True
    return False

# モジュール読み込み時に自動的にActorクラスを拡張
enhance_actor_classes()

# 特別な関数：後からActorクラスを拡張する場合に使用
def enhance_global_actor():
    """グローバルスコープにあるActorクラスを拡張する"""
    return patch_actor_in_globals()
